using UnityEngine;
using UnityEngine.Networking;
using System;
using System.Text;
public class SseDownloadHandler : DownloadHandlerScript
{
    private StringBuilder _lineBuffer = new StringBuilder();
    private string _pendingData = null; // accumulate multi-line data
    public event Action<string> OnMessageReceived; // event for complete SSE messages
                                                   // Constructor: optionally, you can supply a preallocated buffer to base class for efficiency.
    public SseDownloadHandler(int bufferSize = 1024) : base(new byte[bufferSize]) { }

    protected override bool ReceiveData(byte[] data, int dataLength)
    {
        if (data == null || dataLength == 0) return true;
        // Process each byte
        for (int i = 0; i < dataLength; i++)
        {
            char c = (char)data[i];
            if (c == '\r') continue; // ignore carriage return (use \n as standard newline)
            if (c == '\n')
            {
                string line = _lineBuffer.ToString();
                _lineBuffer.Length = 0; // reset buffer for next line
                if (string.IsNullOrEmpty(line))
                {
                    // Blank line indicates end of an event message
                    if (_pendingData != null)
                    {
                        // Complete event received, trigger event callback on main thread
                        OnMessageReceived?.Invoke(_pendingData);
                        _pendingData = null;
                    }
                }
                else if (line.StartsWith("data:"))
                {
                    // Remove the "data:" prefix and any leading space after the colon
                    string dataContent =
                    line.Substring("data:".Length).TrimStart();
                    // Accumulate data lines (in case an event has multiple data: lines)
                    _pendingData = (_pendingData == null) ? dataContent : (_pendingData + "\n" + dataContent);
                }
            }
            // (Optionally handle other SSE fields like "event:" or "id:" here if needed)
            else
            {
                _lineBuffer.Append(c);
            }
        }
    return true; // Return true to indicate we handled the data
    }
}
